# Visualization

## Introduction

Pawpaw included limited visualization tools intended to help develop, test, and debug.

## Format Strings

Pawpaw's ``Ito`` has a well defined  ``.__format__`` that makes it possible to specify a format string for
an ``Ito`` object in [formatted string literals](https://docs.python.org/3/reference/lexical_analysis.html#f-strings) and when using ``str.format()``.
Two categories of formatting directives are available: Integer and String, each with its own vocabulary and modifiers.  Both formatting directives
can occur within a single f-string expression.

### Integer Expressions

An integer expression has the following form:

```
integer expression := directive[:[[fill]align][sign][#][0][width][grouping_option][type]]
```

The *directive* field has one of the following values:

| Directive | Meaning | Example |
| :---:     |  ---    |   ---   |
| ``'%span'`` | ``.span`` as tuple | ``'(1, 14)'`` |
| ``'%start'`` | ``.start`` | ``'1'`` |
| ``'%stop'`` | ``.stop`` | ``'14'`` |

The *fill*, *align*, *sign*, *width*, *grouping_option*, and *type* fields, along with the characters ``'#'`` and ``'0'`` have identical form and behavior as Python's [Format Specification Mini-Language](https://docs.python.org/3/library/string.html?highlight=fill%20align%20sign%20width#format-specification-mini-language). Conversely, the *precision* field and ``'z'`` character[^z_field] are not supported because all ``Ito`` integer directives yield only non-negative integers.

In the case of ``'%span'``, fields are applied to both the values and the result is returned in Python tuple format ``'(x, y)'``.

You may surround integer expressions with additional text as needed, however, if the surrounding text contains characters that could be mis-interpreted as part of the integer expression, you must insert the zero-length whitespace directive, ``'% '`` to allow the parser to distinguish such text from the attached expression.

#### Examples

```python
>>> from pawpaw import Ito
>>> i = Ito(' Hello, world! ', 1, -1, 'phrase')
>>> f'{i:%span}'  # Just .span
f'{i:%span}'  # Just .span
'(1, 14)'
>>> f'{i:%span:~^4}'  # Add some formatting
'(~1~~, ~14~)'
>>> f'{i:[%span:~^4]}'  # Surround with brackets... ok
'[(~1~~, ~14~)]'
>>> f'{i:_%span:~^4_}'  # Now surround with underscores... right underscore will be interpreted as grouping_option
'_(~1~~, ~14~)'
>>> f'{i:_%span:~^4% _}'  # Fix by adding zero-length whitespace directive
'_(~1~~, ~14~)_'
```

### String Expressions

While Pawpaw's Ito integer expressions are nearly identical to Python's, string expressions have many key differences.  A string expresssion has the form:

```
string expression := directive[!conversion][:width[,abbr_suffix]]
```

The *directive* field has one of the following values:

| Directive | Meaning | Example |
| :---:     |  ---    |   ---   |
| ``'%string'`` | ``.string`` | ``' Hello, world! '`` |
| ``'%desc'`` | ``.desc`` | ``'phrase'`` |
| ``'%substr'`` | ``.__str__()`` | ``'Hello, world!'`` |
| ``'%value'`` | ``str(.value())`` | ``'Hello, world!'`` |

The *conversion* field has one of the following three values:

| Value | Meaning             | Example |
| :---: |---------------------|   ---   |
| ``'a'`` | ``ascii``           | ``'\n'`` → ``"'\\n'"`` |
| ``'r'`` | ``repr``            | ``'\n'`` → ``'\n'`` |
| ``'s'`` | ``str`` *(default)* | ``'\n'`` → ``"'\\n'"`` |

The *width* and *abbr_suffix* fields are where Pawpaw significantly differs from Python's format string language.  The *width*
indicates the **_maximum_ width** for the output value, which is truncated as needed.  This is highly uself for large ``Ito`` trees,
in which many nodes will have long substring values. By specifying a maximum width, output can be shortened to faciliate visualization
for interpretation, testing, and debugging purposes.

The *abbr_suffix* field, if present is appended to the truncated value.  If *width* is longer than the length of the *abbr_suffix*,
then *abbr_suffix* itself gets truncated.  Note that you can't use a digit as a value for *abbr_suffix*.  If you try to, it will be
interpreted as part of the decimal value for *width* instead.

As with integer expressions, you may surround your expressions with additonal text as needed, but need to use the zero-length whitespace directive, ``'% '`` to allow the parser to distinguish when such text is not part of the cojoined expression.

Note that if you desired the typical Python min-width behavior, simply use the value (rather than the Ito object) for a normal
Python-style format string:

### Examples:

```python
>>> f'{str(i):~^17}'  # Python format string expression
'~~Hello, world!~~'
>>> f'{i:%substr:4…}'  # Pawpaw format string expression
'Hel…'
>>> f'{i:%span% : %substr}'  # Combine multiple expressions
'(1, 14): Hello, world!'
```

## peduncle

The peduncle[^peduncle] module allows for quick visualization of...

[^z_field]: The ``'z'`` field was added in Python 3.11 and coerces negative zero floating-point values to positive zero after rounding to the format precision  See [PEP 682](https://peps.python.org/pep-0682/) for details.

[^peduncle]: A peduncle is the stalk of a flower.  Pawpaw flowers are maroon in color and born on stout, hairy peduncles.
